<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Auth;
use App\Config;
use App\DB;
use App\Tenant;
use App\View;

final class SaaSController {
  public function showCreateMarketplace(): void {
    // Root domain only
    if (Tenant::current()) View::redirect('/');
    View::render('saas/create-marketplace', []);
  }

  public function createMarketplace(): void {
    if (Tenant::current()) View::redirect('/');
    Auth::checkCsrf();

    $marketName = trim((string)($_POST['market_name'] ?? ''));
    $slug = strtolower(trim((string)($_POST['slug'] ?? '')));
    $ownerName = trim((string)($_POST['owner_name'] ?? ''));
    $ownerEmail = strtolower(trim((string)($_POST['owner_email'] ?? '')));
    $password = (string)($_POST['password'] ?? '');
    $plan = (string)($_POST['plan'] ?? 'basic');

    if ($marketName === '' || $slug === '' || !preg_match('/^[a-z0-9-]{3,30}$/', $slug)) {
      View::flash('error', 'Marketplace name + slug required (3-30 chars: a-z, 0-9, hyphen).');
      View::redirect('/create-marketplace');
    }
    if (!filter_var($ownerEmail, FILTER_VALIDATE_EMAIL) || strlen($password) < 8) {
      View::flash('error', 'Owner email and password (8+ chars) required.');
      View::redirect('/create-marketplace');
    }

    $root = Config::get('APP_ROOT_DOMAIN');
    $exists = DB::one("SELECT id FROM tenants WHERE slug=? LIMIT 1", [$slug]);
    if ($exists) {
      View::flash('error', 'That subdomain is already taken.');
      View::redirect('/create-marketplace');
    }

    $planRow = DB::one("SELECT * FROM plans WHERE code=? LIMIT 1", [$plan]);
    if (!$planRow) $planRow = DB::one("SELECT * FROM plans WHERE code='basic' LIMIT 1", []);

    $tenantId = DB::exec("INSERT INTO tenants (name, slug, plan_id, status) VALUES (?,?,?, 'active')",
      [$marketName, $slug, $planRow['id']]
    );

    // Create admin user for tenant
    $hash = password_hash($password, PASSWORD_BCRYPT);
    $userId = DB::exec("INSERT INTO users (tenant_id, name, email, password_hash, role, email_verified_at)
                        VALUES (?,?,?,?, 'admin', NOW())",
      [$tenantId, $ownerName ?: 'Owner', $ownerEmail, $hash]
    );

    // Seed categories for tenant
    $cats = require __DIR__ . '/../Seeds/categories.php';
    foreach ($cats as $c) {
      DB::exec("INSERT INTO categories (tenant_id, name) VALUES (?,?)", [$tenantId, $c]);
    }

    // Optionally force Stripe checkout for paid plans
    if (Config::bool('STRIPE_ENABLED', false) && ($planRow['price_monthly_cents'] ?? 0) > 0) {
      // Store pending subscription state
      $_SESSION['post_create_tenant_id'] = $tenantId;
      $_SESSION['post_create_user_id'] = $userId;
      $_SESSION['post_create_plan_code'] = $planRow['code'];
      View::redirect('/billing');
    }

    $sub = $slug . '.' . $root;
    View::flash('success', "Marketplace created! Open: https://{$sub}");
    View::redirect('/');
  }

  public function billing(): void {
    // Root domain only
    if (Tenant::current()) View::redirect('/');
    View::render('saas/billing', []);
  }

  public function checkout(): void {
    // Root domain only
    if (Tenant::current()) View::redirect('/');
    Auth::checkCsrf();

    if (!Config::bool('STRIPE_ENABLED', false)) {
      View::flash('error', 'Stripe is disabled. Set STRIPE_ENABLED=true in .env');
      View::redirect('/billing');
    }

    // Minimal Stripe Checkout: this starter only provides placeholders.
    // Implementing full Stripe calls requires stripe-php library (recommended).
    // For cPanel: install via Composer locally and upload vendor/ OR use a VPS.
    View::flash('error', 'Stripe Checkout placeholder. Install stripe-php and wire checkout session creation.');
    View::redirect('/billing');
  }
}
