<?php
declare(strict_types=1);

namespace App\Controllers;

use App\DB;
use App\Tenant;
use App\View;

final class PublicController {
  public function home(): void {
    $tenant = Tenant::current();
    if (!$tenant) {
      // Root marketing
      View::render('saas/marketing', []);
      return;
    }
    // Tenant home: show categories + latest approved
    $cats = DB::q("SELECT * FROM categories WHERE tenant_id = ? ORDER BY name", [$tenant['id']]);
    $ads = DB::q("SELECT a.*, c.name AS category_name
                 FROM ads a
                 JOIN categories c ON c.id = a.category_id
                 WHERE a.tenant_id = ? AND a.status='approved'
                 ORDER BY a.created_at DESC
                 LIMIT 20", [$tenant['id']]);
    View::render('public/tenant-home', compact('cats','ads'));
  }

  public function browse(): void {
    Tenant::requireTenant();
    $tenant = Tenant::current();

    $q = trim((string)($_GET['q'] ?? ''));
    $category = (int)($_GET['category'] ?? 0);
    $state = trim((string)($_GET['state'] ?? ''));
    $city = trim((string)($_GET['city'] ?? ''));
    $min = $_GET['min'] ?? null;
    $max = $_GET['max'] ?? null;

    $params = [$tenant['id']];
    $where = "a.tenant_id = ? AND a.status='approved'";

    if ($q !== '') {
      $where .= " AND (a.title LIKE ? OR a.description LIKE ?)";
      $params[] = "%$q%";
      $params[] = "%$q%";
    }
    if ($category > 0) {
      $where .= " AND a.category_id = ?";
      $params[] = $category;
    }
    if ($state !== '') {
      $where .= " AND a.location_state = ?";
      $params[] = $state;
    }
    if ($city !== '') {
      $where .= " AND a.location_city = ?";
      $params[] = $city;
    }
    if ($min !== null && $min !== '') {
      $where .= " AND a.price >= ?";
      $params[] = (float)$min;
    }
    if ($max !== null && $max !== '') {
      $where .= " AND a.price <= ?";
      $params[] = (float)$max;
    }

    $cats = DB::q("SELECT * FROM categories WHERE tenant_id = ? ORDER BY name", [$tenant['id']]);

    $ads = DB::q("SELECT a.*, c.name AS category_name
                 FROM ads a
                 JOIN categories c ON c.id=a.category_id
                 WHERE $where
                 ORDER BY a.created_at DESC
                 LIMIT 200", $params);

    View::render('public/browse', compact('cats','ads','q','category','state','city','min','max'));
  }

  public function viewAd(): void {
    Tenant::requireTenant();
    $tenant = Tenant::current();
    $id = (int)($_GET['id'] ?? 0);
    $ad = DB::one("SELECT a.*, c.name AS category_name, u.name AS seller_name
                   FROM ads a
                   JOIN categories c ON c.id=a.category_id
                   JOIN users u ON u.id=a.user_id
                   WHERE a.id=? AND a.tenant_id=? AND a.status='approved'
                   LIMIT 1", [$id, $tenant['id']]);
    if (!$ad) {
      http_response_code(404);
      View::render('public/404', []);
      return;
    }
    $images = DB::q("SELECT * FROM ad_images WHERE ad_id=? ORDER BY sort_order ASC, id ASC", [$id]);
    View::render('public/ad', compact('ad','images'));
  }

  public function byState(): void {
    $_GET['state'] = (string)($_GET['state'] ?? '');
    $this->browse();
  }
  public function byCity(): void {
    $_GET['state'] = (string)($_GET['state'] ?? '');
    $_GET['city'] = (string)($_GET['city'] ?? '');
    $this->browse();
  }
}
