<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Auth;
use App\DB;
use App\Tenant;
use App\View;
use App\Helpers\Mailer;

final class AuthController {

  public function showRegister(): void {
    Tenant::requireTenant();
    View::render('auth/register', []);
  }

  public function register(): void {
    Tenant::requireTenant();
    Auth::checkCsrf();
    $tenant = Tenant::current();

    $name = trim((string)($_POST['name'] ?? ''));
    $email = strtolower(trim((string)($_POST['email'] ?? '')));
    $pass = (string)($_POST['password'] ?? '');

    if ($name === '' || $email === '' || strlen($pass) < 8) {
      View::flash('error', 'Please provide name, email, and password (8+ chars).');
      View::redirect('/register');
    }

    $exists = DB::one("SELECT id FROM users WHERE tenant_id=? AND email=? LIMIT 1", [$tenant['id'], $email]);
    if ($exists) {
      View::flash('error', 'That email is already registered.');
      View::redirect('/register');
    }

    // Plan limit: basic example for users count could be enforced here.

    $hash = password_hash($pass, PASSWORD_BCRYPT);
    $uid = DB::exec("INSERT INTO users (tenant_id, name, email, password_hash, role) VALUES (?,?,?,?, 'user')",
      [$tenant['id'], $name, $email, $hash]
    );

    $this->issueVerification($uid, $email);

    View::flash('success', 'Account created. Check your email to verify before posting.');
    View::redirect('/login');
  }

  public function showLogin(): void {
    Tenant::requireTenant();
    View::render('auth/login', []);
  }

  public function login(): void {
    Tenant::requireTenant();
    Auth::checkCsrf();
    $tenant = Tenant::current();

    $email = strtolower(trim((string)($_POST['email'] ?? '')));
    $pass = (string)($_POST['password'] ?? '');

    $u = DB::one("SELECT * FROM users WHERE tenant_id=? AND email=? LIMIT 1", [$tenant['id'], $email]);
    if (!$u || !password_verify($pass, $u['password_hash'])) {
      View::flash('error', 'Invalid login.');
      View::redirect('/login');
    }

    Auth::login($u);
    View::redirect('/dashboard');
  }

  public function logout(): void {
    Auth::checkCsrf();
    Auth::logout();
    View::redirect('/');
  }

  public function verifyEmail(): void {
    Tenant::requireTenant();
    $tenant = Tenant::current();
    $token = (string)($_GET['token'] ?? '');
    if ($token === '') {
      View::redirect('/');
    }

    $rec = DB::one("SELECT ev.*, u.email, u.id AS uid
                    FROM email_verifications ev
                    JOIN users u ON u.id=ev.user_id
                    WHERE ev.token=? AND ev.used_at IS NULL AND ev.expires_at > NOW()
                    AND u.tenant_id=?
                    LIMIT 1", [$token, $tenant['id']]);
    if (!$rec) {
      View::flash('error', 'Verification link invalid or expired.');
      View::redirect('/login');
    }

    DB::exec("UPDATE users SET email_verified_at = NOW() WHERE id=?", [$rec['uid']]);
    DB::exec("UPDATE email_verifications SET used_at = NOW() WHERE id=?", [$rec['id']]);

    // Refresh session if logged in
    $sessionUser = Auth::user();
    if ($sessionUser && (int)$sessionUser['id'] === (int)$rec['uid']) {
      $fresh = DB::one("SELECT * FROM users WHERE id=? LIMIT 1", [$rec['uid']]);
      Auth::login($fresh);
    }

    View::flash('success', 'Email verified! You can now post ads.');
    View::redirect('/dashboard');
  }

  public function resendVerification(): void {
    Auth::checkCsrf();
    Tenant::requireTenant();
    $tenant = Tenant::current();
    $u = Auth::user();

    if (!$u) View::redirect('/login');
    if (!empty($u['email_verified_at'])) {
      View::flash('success', 'Your email is already verified.');
      View::redirect('/dashboard');
    }

    $this->issueVerification((int)$u['id'], (string)$u['email']);
    View::flash('success', 'Verification email re-sent.');
    View::redirect('/dashboard');
  }

  private function issueVerification(int $userId, string $email): void {
    $token = bin2hex(random_bytes(32));
    DB::exec("INSERT INTO email_verifications (user_id, token, expires_at) VALUES (?,?, DATE_ADD(NOW(), INTERVAL 24 HOUR))",
      [$userId, $token]
    );

    $host = $_SERVER['HTTP_HOST'] ?? '';
    $link = (isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $host . "/verify-email?token=" . urlencode($token);

    $html = "<p>Verify your email by clicking below:</p><p><a href='{$link}'>Verify Email</a></p><p>This link expires in 24 hours.</p>";
    Mailer::send($email, "Verify your email", $html);
  }
}
