<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Auth;
use App\DB;
use App\Tenant;
use App\View;
use App\Helpers\Uploads;
use App\Config;

final class AdController {
  public function showPost(): void {
    Tenant::requireTenant();
    Auth::requireVerifiedEmail();

    $tenant = Tenant::current();
    $cats = DB::q("SELECT * FROM categories WHERE tenant_id=? ORDER BY name", [$tenant['id']]);
    View::render('user/post-ad', compact('cats'));
  }

  public function post(): void {
    Tenant::requireTenant();
    Auth::checkCsrf();
    Auth::requireVerifiedEmail();

    $tenant = Tenant::current();
    $u = Auth::user();

    // Plan limit example: ads per month (basic)
    $plan = DB::one("SELECT p.* FROM plans p JOIN tenants t ON t.plan_id=p.id WHERE t.id=? LIMIT 1", [$tenant['id']]);
    $limits = $plan ? json_decode($plan['limits_json'] ?? '{}', true) : [];
    $adsLimit = (int)($limits['ads_per_month'] ?? 50);

    $used = DB::one("SELECT COUNT(*) AS c FROM ads WHERE tenant_id=? AND created_at >= DATE_FORMAT(NOW(), '%Y-%m-01')",
      [$tenant['id']]
    );
    if ($used && (int)$used['c'] >= $adsLimit) {
      View::flash('error', 'Monthly ad limit reached for your marketplace plan.');
      View::redirect('/dashboard');
    }

    $category = (int)($_POST['category_id'] ?? 0);
    $title = trim((string)($_POST['title'] ?? ''));
    $description = trim((string)($_POST['description'] ?? ''));
    $price = (float)($_POST['price'] ?? 0);
    $state = strtoupper(trim((string)($_POST['state'] ?? '')));
    $city = trim((string)($_POST['city'] ?? ''));
    $country = trim((string)($_POST['country'] ?? 'US'));

    if ($category <= 0 || $title === '' || $description === '' || $state === '' || $city === '') {
      View::flash('error', 'Please fill all required fields.');
      View::redirect('/post-ad');
    }

    $adId = DB::exec("INSERT INTO ads (tenant_id, user_id, category_id, title, description, price, location_city, location_state, location_country, status)
                      VALUES (?,?,?,?,?,?,?,?,?, 'pending')",
      [$tenant['id'], $u['id'], $category, $title, $description, $price, $city, $state, $country]
    );

    // Images (multiple)
    $maxImages = (int)($limits['images_per_ad'] ?? 6);

    $files = $_FILES['images'] ?? null;
    if ($files && is_array($files['name'] ?? null)) {
      $count = min(count($files['name']), $maxImages);
      for ($i=0; $i<$count; $i++) {
        $file = [
          'name' => $files['name'][$i] ?? '',
          'type' => $files['type'][$i] ?? '',
          'tmp_name' => $files['tmp_name'][$i] ?? '',
          'error' => $files['error'][$i] ?? UPLOAD_ERR_NO_FILE,
          'size' => $files['size'][$i] ?? 0,
        ];
        if (!Uploads::isValidImage($file)) continue;

        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        $ext = Uploads::extFromMime($mime);

        $dir = __DIR__ . '/../../storage/tenants/' . $tenant['id'] . '/ads/' . $adId;
        if (!is_dir($dir)) mkdir($dir, 0775, true);

        $safe = bin2hex(random_bytes(12)) . '.' . $ext;
        $dest = $dir . '/' . $safe;
        if (move_uploaded_file($file['tmp_name'], $dest)) {
          $publicPath = '/uploads/tenants/' . $tenant['id'] . '/ads/' . $adId . '/' . $safe;
          DB::exec("INSERT INTO ad_images (ad_id, path, sort_order) VALUES (?,?,?)", [$adId, $publicPath, $i]);
        }
      }
    }

    View::flash('success', 'Ad submitted and awaiting admin approval.');
    View::redirect('/dashboard');
  }
}
